// Popup script for Disco Browser

document.addEventListener('DOMContentLoaded', async () => {
  await loadTabs();
  await loadAnalysis();
  
  // Event listeners
  document.getElementById('analyzeBtn').addEventListener('click', analyzeBrowsing);
  document.getElementById('generateBtn').addEventListener('click', showPromptInput);
  document.getElementById('submitPrompt').addEventListener('click', generateApp);
  document.getElementById('optionsBtn').addEventListener('click', openOptions);
});

// Load and display open tabs
async function loadTabs() {
  try {
    const response = await chrome.runtime.sendMessage({ action: 'getTabs' });
    if (response.success) {
      const tabs = response.data.tabs || [];
      displayTabs(tabs);
    }
  } catch (error) {
    console.error('Error loading tabs:', error);
  }
}

// Display tabs in the UI
function displayTabs(tabData) {
  const tabsList = document.getElementById('tabsList');
  const tabCount = document.getElementById('tabCount');
  
  tabCount.textContent = tabData.count || 0;
  
  if (!tabData.tabs || tabData.tabs.length === 0) {
    tabsList.innerHTML = '<div class="empty-state">No tabs open</div>';
    return;
  }
  
  tabsList.innerHTML = tabData.tabs.slice(0, 10).map(tab => `
    <div class="tab-item" title="${tab.title}">
      <img src="${tab.favIconUrl || 'icons/icon16.png'}" alt="">
      <span>${tab.title || 'Untitled'}</span>
    </div>
  `).join('');
}

// Analyze browsing patterns
async function analyzeBrowsing() {
  const analyzeBtn = document.getElementById('analyzeBtn');
  const resultsDiv = document.getElementById('analysisResults');
  
  analyzeBtn.disabled = true;
  analyzeBtn.textContent = 'Analyzing...';
  resultsDiv.innerHTML = '<div class="loading">Analyzing your browsing patterns</div>';
  
  try {
    const response = await chrome.runtime.sendMessage({ action: 'analyze' });
    
    if (response.success) {
      displayAnalysisResults(response.data);
    } else {
      resultsDiv.innerHTML = `<div class="empty-state">Error: ${response.error}</div>`;
    }
  } catch (error) {
    resultsDiv.innerHTML = `<div class="empty-state">Error analyzing: ${error.message}</div>`;
  } finally {
    analyzeBtn.disabled = false;
    analyzeBtn.textContent = 'Analyze';
  }
}

// Display analysis results
function displayAnalysisResults(data) {
  const resultsDiv = document.getElementById('analysisResults');
  const patterns = data.patterns || {};
  
  let html = '';
  
  if (patterns.topDomains && patterns.topDomains.length > 0) {
    html += `
      <div class="pattern-item">
        <h3>Top Domains</h3>
        <div class="domain-list">
          ${patterns.topDomains.map(d => `
            <span class="domain-tag">${d.domain} (${d.count})</span>
          `).join('')}
        </div>
      </div>
    `;
  }
  
  if (data.tabs) {
    html += `
      <div class="pattern-item">
        <h3>Current Session</h3>
        <p>${data.tabs.count} tabs open across ${data.tabs.domains.length} unique domains</p>
      </div>
    `;
  }
  
  if (data.history) {
    html += `
      <div class="pattern-item">
        <h3>Recent History</h3>
        <p>${data.history.count} pages visited in the last 7 days</p>
      </div>
    `;
  }
  
  resultsDiv.innerHTML = html || '<div class="empty-state">No patterns found</div>';
}

// Load cached analysis
async function loadAnalysis() {
  try {
    const result = await chrome.storage.local.get('lastAnalysis');
    if (result.lastAnalysis) {
      displayAnalysisResults(result.lastAnalysis);
    }
  } catch (error) {
    console.error('Error loading cached analysis:', error);
  }
}

// Show prompt input for GenTabs
function showPromptInput() {
  const promptContainer = document.getElementById('promptInput');
  promptContainer.style.display = promptContainer.style.display === 'none' ? 'block' : 'block';
  document.getElementById('userPrompt').focus();
}

// Generate app based on user prompt
async function generateApp() {
  const prompt = document.getElementById('userPrompt').value.trim();
  const generatedApp = document.getElementById('generatedApp');
  const submitBtn = document.getElementById('submitPrompt');
  
  if (!prompt) {
    alert('Please enter a description of what you want to create');
    return;
  }
  
  submitBtn.disabled = true;
  submitBtn.textContent = 'Generating...';
  generatedApp.classList.add('active');
  generatedApp.innerHTML = '<div class="loading">Creating your personalized app</div>';
  
  try {
    // Get browsing context
    const response = await chrome.runtime.sendMessage({ action: 'analyze' });
    const context = response.success ? response.data : null;
    
    // Generate app using AI service
    const appData = await generateAppWithAI(prompt, context);
    
    displayGeneratedApp(appData);
  } catch (error) {
    generatedApp.innerHTML = `<div class="empty-state">Error: ${error.message}</div>`;
  } finally {
    submitBtn.disabled = false;
    submitBtn.textContent = 'Create';
  }
}

// Generate app using AI service
const aiService = new AIService();

async function generateAppWithAI(prompt, context) {
  return await aiService.generateApp(prompt, context);
}

// Display generated app
function displayGeneratedApp(appData) {
  const generatedApp = document.getElementById('generatedApp');
  
  generatedApp.innerHTML = `
    <h3>${appData.title}</h3>
    <p>${appData.description}</p>
    ${appData.html}
  `;
}

// Open options page
function openOptions() {
  chrome.runtime.openOptionsPage();
}

