#!/usr/bin/env python3
"""Extract dominant colors from disco.webp"""

import sys
import os

try:
    from PIL import Image
    import colorsys
except ImportError:
    print("PIL/Pillow not installed. Installing...")
    os.system("pip3 install Pillow --quiet")
    from PIL import Image
    import colorsys

def rgb_to_hex(r, g, b):
    return f"#{r:02x}{g:02x}{b:02x}"

def extract_colors(image_path, num_colors=10):
    """Extract dominant colors from image"""
    try:
        img = Image.open(image_path)
        img = img.convert('RGB')
        
        # Resize for faster processing
        img.thumbnail((200, 200))
        
        # Get all pixels
        pixels = list(img.getdata())
        
        # Count color frequencies (quantized)
        color_map = {}
        for r, g, b in pixels:
            # Quantize to reduce variations
            qr = (r // 10) * 10
            qg = (g // 10) * 10
            qb = (b // 10) * 10
            key = (qr, qg, qb)
            color_map[key] = color_map.get(key, 0) + 1
        
        # Sort by frequency
        sorted_colors = sorted(color_map.items(), key=lambda x: x[1], reverse=True)[:num_colors]
        
        return sorted_colors
    except Exception as e:
        print(f"Error: {e}")
        return []

def main():
    image_path = "disco.webp"
    
    if not os.path.exists(image_path):
        print(f"Error: {image_path} not found!")
        return
    
    print("\n=== Extracting colors from disco.webp ===\n")
    
    colors = extract_colors(image_path, 10)
    
    if not colors:
        print("Failed to extract colors")
        return
    
    print("Top Colors:")
    hex_colors = []
    for i, ((r, g, b), count) in enumerate(colors, 1):
        hex_color = rgb_to_hex(r, g, b)
        hex_colors.append(hex_color)
        print(f"{i}. {hex_color} (rgb({r}, {g}, {b})) - {count} occurrences")
    
    print("\n=== CSS Variables ===\n")
    print(":root {")
    for i, hex_color in enumerate(hex_colors, 1):
        print(f"  --disco-color-{i}: {hex_color};")
    print("}\n")
    
    print("=== Gradient ===\n")
    primary = hex_colors[0]
    secondary = hex_colors[1] if len(hex_colors) > 1 else hex_colors[0]
    accent = hex_colors[2] if len(hex_colors) > 2 else hex_colors[1] if len(hex_colors) > 1 else hex_colors[0]
    
    print(f"Primary: {primary}")
    print(f"Secondary: {secondary}")
    print(f"Accent: {accent}\n")
    
    print("CSS Gradient:")
    print(f"background: linear-gradient(135deg, {primary} 0%, {secondary} 50%, {accent} 100%);\n")
    
    # Save to file
    with open("extracted-colors.txt", "w") as f:
        f.write("Extracted Colors from disco.webp\n")
        f.write("=" * 40 + "\n\n")
        f.write("CSS Variables:\n")
        f.write(":root {\n")
        for i, hex_color in enumerate(hex_colors, 1):
            f.write(f"  --disco-color-{i}: {hex_color};\n")
        f.write("}\n\n")
        f.write(f"Primary: {primary}\n")
        f.write(f"Secondary: {secondary}\n")
        f.write(f"Accent: {accent}\n")
    
    print("Colors saved to extracted-colors.txt")
    
    return hex_colors

if __name__ == "__main__":
    main()

