// Node.js script to extract colors from disco.webp
// Run with: node extract-colors.js

const fs = require('fs');
const path = require('path');

// Try to use sharp if available, otherwise provide instructions
try {
  const sharp = require('sharp');
  
  async function extractColors() {
    const imagePath = path.join(__dirname, 'disco.webp');
    
    if (!fs.existsSync(imagePath)) {
      console.log('disco.webp not found!');
      return;
    }
    
    const image = sharp(imagePath);
    const { data, info } = await image.raw().toBuffer({ resolveWithObject: true });
    
    // Sample pixels
    const colorMap = {};
    const sampleRate = 100;
    
    for (let i = 0; i < data.length; i += sampleRate * info.channels) {
      const r = data[i];
      const g = data[i + 1];
      const b = data[i + 2];
      
      // Quantize
      const qr = Math.floor(r / 10) * 10;
      const qg = Math.floor(g / 10) * 10;
      const qb = Math.floor(b / 10) * 10;
      const key = `${qr},${qg},${qb}`;
      
      colorMap[key] = (colorMap[key] || 0) + 1;
    }
    
    // Get top colors
    const sortedColors = Object.entries(colorMap)
      .sort((a, b) => b[1] - a[1])
      .slice(0, 10);
    
    console.log('\n=== Top Colors from disco.webp ===\n');
    
    const hexColors = sortedColors.map(([rgb]) => {
      const [r, g, b] = rgb.split(',').map(Number);
      return rgbToHex(r, g, b);
    });
    
    sortedColors.forEach(([rgb, count], index) => {
      const [r, g, b] = rgb.split(',').map(Number);
      const hex = rgbToHex(r, g, b);
      console.log(`${index + 1}. ${hex} (rgb(${r},${g},${b})) - ${count} occurrences`);
    });
    
    console.log('\n=== CSS Variables ===\n');
    console.log(':root {');
    hexColors.forEach((hex, index) => {
      console.log(`  --disco-color-${index + 1}: ${hex};`);
    });
    console.log('}\n');
    
    console.log('=== Gradient ===\n');
    console.log(`background: linear-gradient(135deg, ${hexColors[0]} 0%, ${hexColors[1] || hexColors[0]} 50%, ${hexColors[2] || hexColors[1] || hexColors[0]} 100%);\n`);
    
    return hexColors;
  }
  
  function rgbToHex(r, g, b) {
    return "#" + [r, g, b].map(x => {
      const hex = x.toString(16);
      return hex.length === 1 ? "0" + hex : hex;
    }).join("");
  }
  
  extractColors().catch(console.error);
  
} catch (e) {
  console.log('Sharp not available. Please install: npm install sharp');
  console.log('Or open extract-colors.html in your browser to extract colors.');
}

