// Full-screen Disco interface script

const aiService = new AIService();
const themeManager = new ThemeManager();

document.addEventListener('DOMContentLoaded', async () => {
  // Initialize theme
  await themeManager.loadTheme();
  
  await loadTabs();
  await loadAnalysis();
  
  // Event listeners
  document.getElementById('analyzeBtn').addEventListener('click', analyzeBrowsing);
  document.getElementById('generateBtn').addEventListener('click', generateApp);
  document.getElementById('closeBtn').addEventListener('click', closeDisco);
  document.getElementById('settingsBtn').addEventListener('click', openSettings);
  document.getElementById('themeBtn').addEventListener('click', toggleThemeMenu);
  
  // Initialize theme menu
  initThemeMenu();
  
  // Auto-focus prompt input
  document.getElementById('userPrompt').focus();
  
  // Close theme menu when clicking outside
  document.addEventListener('click', (e) => {
    const themeMenu = document.getElementById('themeMenu');
    const themeBtn = document.getElementById('themeBtn');
    if (!themeMenu.contains(e.target) && !themeBtn.contains(e.target)) {
      themeMenu.classList.add('hidden');
    }
  });
});

// Load and display open tabs
async function loadTabs() {
  try {
    const response = await chrome.runtime.sendMessage({ action: 'getTabs' });
    if (response.success) {
      const tabs = response.data.tabs || [];
      displayTabs(tabs);
    }
  } catch (error) {
    console.error('Error loading tabs:', error);
  }
}

// Display tabs in the UI
function displayTabs(tabData) {
  const tabsList = document.getElementById('tabsList');
  const tabCount = document.getElementById('tabCount');
  
  tabCount.textContent = tabData.count || 0;
  
  if (!tabData.tabs || tabData.tabs.length === 0) {
    tabsList.innerHTML = '<div class="empty-state">No tabs open</div>';
    return;
  }
  
  tabsList.innerHTML = tabData.tabs.map(tab => `
    <div class="tab-item" data-url="${tab.url}" title="${tab.title}">
      <img src="${tab.favIconUrl || 'icons/icon16.png'}" alt="" onerror="this.src='icons/icon16.png'">
      <span>${tab.title || 'Untitled'}</span>
    </div>
  `).join('');
  
  // Add click handlers to tabs
  tabsList.querySelectorAll('.tab-item').forEach(item => {
    item.addEventListener('click', () => {
      const url = item.getAttribute('data-url');
      if (url) {
        chrome.tabs.create({ url });
      }
    });
  });
}

// Analyze browsing patterns
async function analyzeBrowsing() {
  const analyzeBtn = document.getElementById('analyzeBtn');
  const resultsDiv = document.getElementById('analysisResults');
  
  analyzeBtn.disabled = true;
  analyzeBtn.innerHTML = '<span>⏳ Analyzing...</span>';
  
  try {
    const response = await chrome.runtime.sendMessage({ action: 'analyze' });
    
    if (response.success) {
      displayAnalysisResults(response.data);
    } else {
      resultsDiv.innerHTML = `<div class="insight-card"><div class="insight-icon">⚠️</div><div class="insight-text"><h3>Error</h3><p>${response.error}</p></div></div>`;
    }
  } catch (error) {
    resultsDiv.innerHTML = `<div class="insight-card"><div class="insight-icon">⚠️</div><div class="insight-text"><h3>Error</h3><p>${error.message}</p></div></div>`;
  } finally {
    analyzeBtn.disabled = false;
    analyzeBtn.innerHTML = '<span>📊 Analyze Browsing</span>';
  }
}

// Display analysis results
function displayAnalysisResults(data) {
  const resultsDiv = document.getElementById('analysisResults');
  const patterns = data.patterns || {};
  
  let html = '';
  
  // Top Domains
  if (patterns.topDomains && patterns.topDomains.length > 0) {
    html += `
      <div class="insight-card">
        <div class="insight-icon">🌐</div>
        <div class="insight-text">
          <h3>Top Domains</h3>
          <div class="domain-list">
            ${patterns.topDomains.slice(0, 5).map(d => `
              <span class="domain-tag">${d.domain} (${d.count})</span>
            `).join('')}
          </div>
        </div>
      </div>
    `;
  }
  
  // Current Session
  if (data.tabs) {
    html += `
      <div class="insight-card">
        <div class="insight-icon">📑</div>
        <div class="insight-text">
          <h3>Current Session</h3>
          <p>${data.tabs.count} tabs open across ${data.tabs.domains.length} unique domains</p>
        </div>
      </div>
    `;
  }
  
  // Recent History
  if (data.history) {
    html += `
      <div class="insight-card">
        <div class="insight-icon">📚</div>
        <div class="insight-text">
          <h3>Recent History</h3>
          <p>${data.history.count} pages visited in the last 7 days</p>
        </div>
      </div>
    `;
  }
  
  resultsDiv.innerHTML = html || '<div class="empty-state">No patterns found</div>';
}

// Load cached analysis
async function loadAnalysis() {
  try {
    const result = await chrome.storage.local.get('lastAnalysis');
    if (result.lastAnalysis) {
      displayAnalysisResults(result.lastAnalysis);
    }
  } catch (error) {
    console.error('Error loading cached analysis:', error);
  }
}

// Generate app based on user prompt
async function generateApp() {
  const prompt = document.getElementById('userPrompt').value.trim();
  const generatedApp = document.getElementById('generatedApp');
  const generateBtn = document.getElementById('generateBtn');
  
  if (!prompt) {
    alert('Please enter a description of what you want to create');
    return;
  }
  
  generateBtn.disabled = true;
  generateBtn.innerHTML = '<span>✨ Generating...</span>';
  generatedApp.classList.add('active');
  generatedApp.innerHTML = '<div class="loading">Creating your personalized app</div>';
  
  try {
    // Get browsing context
    const response = await chrome.runtime.sendMessage({ action: 'analyze' });
    const context = response.success ? response.data : null;
    
    // Generate app using AI service
    const appData = await aiService.generateApp(prompt, context);
    
    displayGeneratedApp(appData);
  } catch (error) {
    generatedApp.innerHTML = `<div class="empty-state">Error: ${error.message}</div>`;
  } finally {
    generateBtn.disabled = false;
    generateBtn.innerHTML = '<span>✨ Generate App</span>';
  }
}

// Display generated app
function displayGeneratedApp(appData) {
  const generatedApp = document.getElementById('generatedApp');
  
  generatedApp.innerHTML = `
    <h3>${appData.title}</h3>
    <p>${appData.description}</p>
    ${appData.html}
  `;
}

// Close disco interface
function closeDisco() {
  window.close();
}

// Open settings
function openSettings() {
  chrome.runtime.openOptionsPage();
}

// Theme menu functions
function initThemeMenu() {
  const themeList = document.getElementById('themeList');
  const themes = themeManager.getThemes();
  const currentTheme = themeManager.getCurrentTheme();
  
  themeList.innerHTML = Object.entries(themes).map(([key, theme]) => `
    <div class="theme-item ${key === currentTheme ? 'active' : ''}" data-theme="${key}">
      <div class="theme-preview" style="background: ${theme.colors.background}"></div>
      <div class="theme-info">
        <div class="theme-name">${theme.name}</div>
        <div class="theme-desc">${theme.description}</div>
      </div>
    </div>
  `).join('');
  
  // Add click handlers
  themeList.querySelectorAll('.theme-item').forEach(item => {
    item.addEventListener('click', () => {
      const themeKey = item.getAttribute('data-theme');
      themeManager.saveTheme(themeKey);
      updateThemeMenu();
    });
  });
}

function toggleThemeMenu() {
  const themeMenu = document.getElementById('themeMenu');
  themeMenu.classList.toggle('hidden');
}

function updateThemeMenu() {
  const currentTheme = themeManager.getCurrentTheme();
  document.querySelectorAll('.theme-item').forEach(item => {
    if (item.getAttribute('data-theme') === currentTheme) {
      item.classList.add('active');
    } else {
      item.classList.remove('active');
    }
  });
}

// Keyboard shortcuts
document.addEventListener('keydown', (e) => {
  // ESC to close
  if (e.key === 'Escape') {
    const themeMenu = document.getElementById('themeMenu');
    if (!themeMenu.classList.contains('hidden')) {
      themeMenu.classList.add('hidden');
    } else {
      closeDisco();
    }
  }
  
  // Cmd/Ctrl + Enter to generate
  if ((e.metaKey || e.ctrlKey) && e.key === 'Enter') {
    e.preventDefault();
    generateApp();
  }
});

